import React, { useState, useEffect } from "react";

// Componente reutilizable para Input de Texto
const TextInput = ({ label, value, onChange, placeholder, error }) => (
  <div className="mb-4">
    <label className="block text-base font-bold text-gray-700 ml-2">
      {label}
    </label>
    <input
      className={`w-80 px-2 py-2 border rounded-md focus:outline-none text-black ${
        error ? "focus:border-red-500" : "focus:border-blue-500"
      }`}
      type="text"
      value={value}
      onChange={onChange}
      placeholder={placeholder}
    />
    {error && <p className="text-red-500 text-sm mt-1 ml-1">{error}</p>}
  </div>
);

// Componente reutilizable para Textarea
const TextArea = ({ label, value, onChange, placeholder, error }) => (
  <div className="mb-4">
    <label className="block text-base font-bold text-gray-700 ml-2">
      {label}
    </label>
    <textarea
      className={`w-full px-2 py-2 border rounded-md focus:outline-none text-black ${
        error ? "focus:border-red-500" : "focus:border-blue-500"
      }`}
      value={value}
      onChange={onChange}
      placeholder={placeholder}
      rows="4"
      style={{ resize: "none" }}
    />
    {error && <p className="text-red-500 text-sm mt-1 ml-1">{error}</p>}
  </div>
);

export default function EditarModal({ empresaSeleccionada, onSave, onClose }) {
  const [formData, setFormData] = useState({
    nombreEmpresa: "",
    rut: "",
    telefono: "",
    representanteLegal: "",
    nombreEjecutivo: "",
    tipoSociedad: "",
    direccionE: "",
    correoElectronico: "",
    presentacionE: "",
    serviciosE: "",
    visionE: "",
    misionE: "",
    errors: {
      nombreEmpresa: null,
      rut: null,
      telefono: null,
      representanteLegal: null,
      nombreEjecutivo: null,
      tipoSociedad: null,
      direccionE: null,
      correoElectronico: null,
      presentacionE: null,
      serviciosE: null,
      visionE: null,
      misionE: null,
    },
  });

  const resetErrors = () => {
    const emptyErrors = Object.fromEntries(
      Object.keys(formData.errors).map((key) => [key, null])
    );
    setFormData((prevFormData) => ({
      ...prevFormData,
      errors: emptyErrors,
    }));
  };

  useEffect(() => {
    // Resetea los errores al abrir el modal
    resetErrors();
  }, []);

  const formatRut = (rut, maxLength) => {
    rut = rut.slice(0, maxLength);
    rut = rut.replace(/[^\dkK]/g, "");
    if (rut.length > 1) {
      rut = rut.replace(/^(\d{1,2})(\d{3})(\d{3})/, "$1.$2.$3");
      rut = rut.replace(/^(\d{2}\.\d{3}\.\d{3})/, "$1-");
    }
    return rut;
  };

  const handleOnChange = (e, field, maxLength) => {
    let value = e.target.value;
    let error = null;

    switch (field) {
      case "rut":
        value = formatRut(value, maxLength);
        if (
          e.nativeEvent.inputType === "deleteContentBackward" &&
          value.slice(-1) === "-"
        ) {
          value = value.slice(0, -2);
        }
        error = !/^(\d{1,2}\.\d{3}\.\d{3}-[0-9kK])$/.test(value)
          ? "Formato de RUT no válido"
          : null;
        break;

      case "telefono":
        value = value.replace(/\D/g, "").slice(0, maxLength);
        break;

      case "correoElectronico":
        value = value.replace(/[^\w@.-]/g, "").slice(0, maxLength);
        break;

      default:
        value = value
          .replace(/[^a-zA-ZáéíóúÁÉÍÓÚ\sñÑ.,]/g, "")
          .slice(0, maxLength);
        break;
    }

    setFormData((prevFormData) => ({
      ...prevFormData,
      [field]: value,
      errors: { ...prevFormData.errors, [field]: error },
    }));
  };

  const validateForm = () => {
    const fields = Object.keys(formData);

    const updatedErrors = fields.reduce((errors, field) => {
      let error = null;

      switch (field) {
        case "rut":
          error = !/^(\d{1,2}\.\d{3}\.\d{3}-[0-9kK])$/.test(formData[field])
            ? "Formato de RUT no válido"
            : null;
          break;

        case "telefono":
          error = !/^\d{9}$/.test(formData[field])
            ? "Formato de teléfono no válido"
            : null;
          break;

        case "correoElectronico":
          error = !/^[\w-]+(\.[\w-]+)*@([\w-]+\.)+[a-zA-Z]{2,7}$/.test(
            formData[field]
          )
            ? "Formato de correo electrónico no válido"
            : null;
          break;

        default:
          const trimmedValue =
            typeof formData[field] === "string"
              ? formData[field].trim()
              : formData[field];
          error = trimmedValue.length === 0 ? "Campo obligatorio" : null;
          break;
      }

      return { ...errors, [field]: error };
    }, {});

    setFormData((prevFormData) => ({
      ...prevFormData,
      errors: updatedErrors,
    }));

    return Object.values(updatedErrors).every((error) => error === null);
  };

  const handleSave = () => {
    if (!validateForm()) {
      return;
    }
    // Actualizar el estado en el componente padre
    onSave({
      empresa_id: empresaSeleccionada.empresa_id,
      nombre_empresa: formData.nombreEmpresa,
      representante_legal: formData.representanteLegal,
      rut_empresa: formData.rut,
      direccion_emp: formData.direccionE,
      tipo_sociedad: formData.tipoSociedad,
      nombre_ejecutivo: formData.nombreEjecutivo,
      telefono_ejecutivo: formData.telefono,
      correo_ejecutivo: formData.correoElectronico,
      presentacion_empresa: formData.presentacionE,
      vision_empresa: formData.visionE,
      mision_empresa: formData.misionE,
      descripcion_servicios: formData.serviciosE,
    });

    // Limpiar el formulario después de guardar
    setFormData({
      nombreEmpresa: "",
      rut: "",
      telefono: "",
      representanteLegal: "",
      nombreEjecutivo: "",
      tipoSociedad: "",
      direccionE: "",
      correoElectronico: "",
      presentacionE: "",
      serviciosE: "",
      visionE: "",
      misionE: "",
      errors: "",
    });
    // Cerrar el modal
    onClose();
  };

  useEffect(() => {
    if (empresaSeleccionada) {
      setFormData({
        rut: formatRut(empresaSeleccionada.rut_empresa || ""),
        telefono: empresaSeleccionada.telefono_ejecutivo || "",
        nombreEmpresa: empresaSeleccionada.nombre_empresa || "",
        representanteLegal: empresaSeleccionada.representante_legal || "",
        nombreEjecutivo: empresaSeleccionada.nombre_ejecutivo || "",
        tipoSociedad: empresaSeleccionada.tipo_sociedad || "",
        direccionE: empresaSeleccionada.direccion_emp || "",
        correoElectronico: empresaSeleccionada.correo_ejecutivo || "",
        presentacionE: empresaSeleccionada.presentacion_empresa || "",
        serviciosE: empresaSeleccionada.descripcion_servicios || "",
        visionE: empresaSeleccionada.vision_empresa || "",
        misionE: empresaSeleccionada.mision_empresa || "",
        errors: {},
      });
    }
  }, [empresaSeleccionada]);

  return (
    <div>
      <div className="w-1/2 border-b border-black mb-4">
        <h1 className="text-black text-2xl font-bold mb-[1px] ml-1">
          Editar Empresa
        </h1>
      </div>
      <div className="grid grid-cols-3 gap-2 text-black">
        <div>
          <TextInput
            label="Nombre de la Empresa:"
            id="nombreEmpresa"
            value={formData.nombreEmpresa}
            onChange={(e) => handleOnChange(e, "nombreEmpresa", 20)}
            placeholder="Ingrese el nombre"
            error={formData.errors.nombreEmpresa}
          />
          <TextInput
            label="Tipo de Sociedad:"
            id="tipoSociedad"
            value={formData.tipoSociedad}
            onChange={(e) => handleOnChange(e, "tipoSociedad", 40)}
            placeholder="Ingrese tipo sociedad"
            error={formData.errors.tipoSociedad}
          />
          <TextInput
            label="Correo Ejecutivo:"
            id="correoEjecutivo"
            value={formData.correoElectronico}
            onChange={(e) => handleOnChange(e, "correoElectronico", 40)}
            placeholder="Ingrese Correo del ejecutivo"
            error={formData.errors.correoElectronico}
          />
        </div>
        <div>
          <TextInput
            label="Rut de la Empresa:"
            id="rutEmpresa"
            value={formData.rut}
            onChange={(e) => handleOnChange(e, "rut", 12)}
            placeholder="Ingrese RUT de la empresa"
            error={formData.errors.rut}
          />
          <TextInput
            label="Representante Legal:"
            id="representanteLegal"
            value={formData.representanteLegal}
            onChange={(e) => handleOnChange(e, "representanteLegal", 20)}
            placeholder="Ingrese Representante Legal"
            error={formData.errors.representanteLegal}
          />
          <TextInput
            label="Telefono de Ejecutivo:"
            id="telEjecutivo"
            value={formData.telefono}
            onChange={(e) => handleOnChange(e, "telefono", 9)}
            placeholder="Ingrese tipo sociedad"
            error={formData.errors.telefono}
          />
        </div>
        <div>
          <TextInput
            label="Dirección de la Empresa:"
            id="direccionEmpresa"
            value={formData.direccionE}
            onChange={(e) => handleOnChange(e, "direccionE", 30)}
            placeholder="Ingrese dirección empresa"
            error={formData.errors.direccionE}
          />
          <TextInput
            label="Nombre de ejecutivo:"
            id="nombreEjecutivo"
            value={formData.nombreEjecutivo}
            onChange={(e) => handleOnChange(e, "nombreEjecutivo", 20)}
            placeholder="Ingrese tipo sociedad"
            error={formData.errors.nombreEjecutivo}
          />
        </div>
      </div>

      <div className="grid grid-cols-2 gap-4 text-black">
        <div>
          <TextArea
            label="Presetanción de la Empresa:"
            id="presentacion"
            rows="2"
            style={{ resize: "none" }}
            value={formData.presentacionE}
            onChange={(e) => handleOnChange(e, "presentacionE", 100)}
            placeholder="Ingrese presentación de la empresa"
            error={formData.errors.presentacionE}
          />

          <TextArea
            label="Visión de la Empresa:"
            id="vision"
            rows="2"
            style={{ resize: "none" }}
            value={formData.visionE}
            onChange={(e) => handleOnChange(e, "visionE", 100)}
            placeholder="Ingrese visión de la empresa"
            error={formData.errors.visionE}
          />
        </div>
        <div>
          {/* Segunda columna */}
          <TextArea
            label="Servicios de la Empresa:"
            id="servicios"
            rows="2"
            style={{ resize: "none" }}
            value={formData.serviciosE}
            onChange={(e) => handleOnChange(e, "serviciosE", 100)}
            placeholder="Ingrese Correo del ejecutivo"
            error={formData.errors.serviciosE}
          />

          <TextArea
            label="Misión de la Empresa:"
            id="mision"
            rows="2"
            style={{ resize: "none" }}
            value={formData.misionE}
            onChange={(e) => handleOnChange(e, "misionE", 100)}
            placeholder="Ingrese Correo del ejecutivo"
            error={formData.errors.misionE}
          />
        </div>
      </div>
      <div className="mt-4 text-right">
        <button
          className="bg-blue-500 hover:bg-blue-600 text-white px-4 py-2 rounded-lg mr-2"
          onClick={handleSave}
        >
          Guardar
        </button>
        <button
          className="bg-gray-500 hover:bg-gray-600 text-white px-4 py-2 rounded-lg"
          onClick={onClose}
        >
          Cerrar
        </button>
      </div>
    </div>
  );
}
