"use client";
import React, { useState, useEffect, useMemo } from "react";
import {
  BuildingOffice2Icon,
  DocumentIcon,
  PencilIcon,
  MagnifyingGlassIcon,
  UserCircleIcon,
} from "@heroicons/react/24/solid";
import Modal from "../../../components/admin/Modal.jsx";
import AgregarModal from "./modalagregar.jsx";
import DetallesModal from "./modaldetalles.jsx";
import EditarModal from "./modaleditar.jsx";
import UsersModal from "./modalusers.jsx";
import { getEmpresas } from "@/api/empresas.jsx";
import { useSession } from "@/app/contexts/SessionContext.jsx";
import Link from "next/link.js";

// Tabla y modal
export default function AdminEmpresas() {

  const { user } = useSession();

  const [isModalOpen, setIsModalOpen] = useState(false);
  const [modalAction, setModalAction] = useState("");
  const [empresas, setEmpresas] = useState([]);
  const [empresaSeleccionada, setEmpresaSeleccionada] = useState(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [rowStates, setRowStates] = useState({});
  const [successMessage, setSuccessMessage] = useState({
    message: "",
    visible: false,
  });

  useEffect(() => {
    // Inicializar el estado local de cada fila
    const initialRowStates = {};
    empresas.forEach((empresa) => {
      initialRowStates[empresa.empresa_id] = false; // Puedes ajustar esto según tu lógica inicial
    });
    setRowStates(initialRowStates);
  }, [empresas]);

  const handleRowClick = (empresaId) => {
    setRowStates((prevRowStates) => {
      const newRowStates = {
        ...prevRowStates,
        [empresaId]: !prevRowStates[empresaId],
      };
      return newRowStates;
    });
  };

  useEffect(() => {
    const fetchEmpresas = async () => {
      try {
        if (user && user.token) {
          const data = await getEmpresas(user.token);
          setEmpresas(data);
          console.log(data)
        }
      } catch (error) {
        console.error('Error al obtener las empresas:', error.message);
      }
    };

    fetchEmpresas();
  }, [user]);



  const openModal = (action, empresa) => {
    setIsModalOpen(true);
    setModalAction(action);
    setEmpresaSeleccionada(empresa);
  };

  const closeModal = () => {
    setIsModalOpen(false);
    setEmpresaSeleccionada(null);
  };

  const handleSaveEmpresa = (empresaActualizada) => {
    // Actualizar la empresa en el estado
    setEmpresas((prevEmpresas) => {
      const updatedEmpresas = prevEmpresas.map((empresa) =>
        empresa.empresa_id === empresaActualizada.empresa_id
          ? { ...empresa, ...empresaActualizada }
          : empresa
      );
      return updatedEmpresas;
    });
    // Actualizar el estado del botón booleano
    setRowStates((prevRowStates) => {
      const updatedRowStates = { ...prevRowStates };
      updatedRowStates[empresaActualizada.empresa_id] = true;
      return updatedRowStates;
    });

    closeModal();
    setSuccessMessage({
      message: "¡La empresa se ha guardado correctamente!",
      visible: true,
    });

    // Configurar el temporizador para ocultar el mensaje después de 3 segundos
    setTimeout(() => {
      setSuccessMessage({ ...successMessage, visible: false });
    }, 3000);
  };

  const handleAddEmpresa = (nuevaEmpresa) => {
    // Agregar la nueva empresa al estado
    setEmpresas((prevEmpresas) => [...prevEmpresas, nuevaEmpresa]);
    setSuccessMessage({
      message: "¡La nueva empresa se ha añadido correctamente!",
      visible: true,
    });

    // Configurar el temporizador para ocultar el mensaje después de 3 segundos
    setTimeout(() => {
      setSuccessMessage({ ...successMessage, visible: false });
    }, 3000);
  };

  const filteredEmpresas = useMemo(() => {
    return empresas.filter((empresa) => {
      if (!empresa || !empresa.nombre_empresa) {
        return false;
      }

      const searchTermLowerCase = searchTerm.toLowerCase();

      return (
        empresa.nombre_empresa.toLowerCase().includes(searchTermLowerCase) ||
        empresa.empresa_id.toString().includes(searchTermLowerCase) ||
        empresa.rut_empresa.includes(searchTermLowerCase)
      );
    });
  }, [empresas, searchTerm]);

  const modalComponents = {
    agregar: (
      <AgregarModal
        onSave={(nuevaEmpresa) => {
          handleAddEmpresa(nuevaEmpresa);
          handleSaveEmpresa(nuevaEmpresa);
        }}
        onClose={closeModal}
      />
    ),
    editar: empresaSeleccionada && (
      <EditarModal
        empresaSeleccionada={empresaSeleccionada}
        onSave={handleSaveEmpresa}
        onClose={closeModal}
      />
    ),
    detalles: empresaSeleccionada && (
      <DetallesModal
        empresaSeleccionada={empresaSeleccionada}
        onClose={closeModal}
      />
    ),
    usuarios: empresaSeleccionada && (
      <UsersModal
        empresaSeleccionada={empresaSeleccionada}
        onClose={closeModal}
      />
    ),
  };

  const modalContent = modalComponents[modalAction] || null;

  return (
    <div className="flex flex-col flex-1 max-w-full">
      <div className="mb-2 max-w-full sm:max-w-sm  md:max-w-md lg:max-w-lg xl:max-w-xl">
        <h1 className="flex items-center text-3xl font-bold text-black border-b border-black pb-2">
          <BuildingOffice2Icon className="h-8 w-8 mr-1" /> Empresas
        </h1>
      </div>

      <div className="flex justify-between mt-4">
        <button
          title="Agregar nueva empresa"
          onClick={() => openModal("agregar")}
          className="text-white bg-sky-500 hover:bg-sky-600 rounded-lg px-5 py-2 shadow drop-shadow-md"
        >
          Agregar Empresa
        </button>
        <div className="relative">
          <MagnifyingGlassIcon className="absolute text-black left-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-gray-400" />
          <input
            type="search"
            className="text-black rounded-lg pl-10 border border-gray-400 py-2 px-3"
            placeholder="Buscar..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)} // Actualizar el valor de búsqueda al escribir en el input
          />
        </div>
      </div>
      <div>
        <table className="mt-1 table-auto w-full rounded-lg overflow-hidden shadow-xs">
          <thead>
            <tr className="bg-gray-400 text-black border-b border-gray-500">
              <th className="px-6 py-2">Abreviación</th>
              <th className="px-6 py-2">Nombre</th>
              <th className="px-6 py-2">Ejecutivo</th>
              <th className="px-6 py-2">Teléfono</th>
              <th className="px-6 py-2">Estado</th>
              <th className="px-6 py-2">Postulaciones</th>
              <th className="px-6 py-2">Rubro</th>
              <th className="px-6 py-2">Detalles</th>
            </tr>
          </thead>
          {/* Contenido de la tabla */}
          <tbody>
            {filteredEmpresas.map((empresa) => (
              <tr
                key={empresa.empresa_id}
                className="border-collapse text-black bg-gray-300 border-b text-center border-gray-500"
              >
                <td className="py-2 px-4 text-base">{empresa.abreviacion}</td>
                <td className="py-2 px-4 text-sm">{empresa.nombre_empresa}</td>
                <td className="py-2 px-4 text-sm">{empresa.nombre_ejecutivo}</td>
                <td className="py-2 px-4 text-sm">{empresa.telefono_ejecutivo}</td>

                <td className="py-2 px-4">
                  <input
                    className={`h-3.5 w-8  appearance-none rounded-[0.4375rem] bg-neutral-300 before:pointer-events-none before:absolute before:h-3.5 before:w-3.5 before:rounded-full before:bg-transparent before:content-[''] after:absolute after:z-[2] after:-mt-[0.1875rem] after:h-5 after:w-5  after:ml-[-0.2rem] after:rounded-full after:border-none after:bg-neutral-100 after:shadow-[0_0px_3px_0_rgb(0_0_0/_7%),_0_2px_2px_0_rgb(0_0_0/_4%)] after:transition-[background-color_0.2s,transform_0.2s] after:content-[''] checked:bg-blue-500 checked:after:absolute checked:after:z-[2] checked:after:-mt-[3px] checked:after:ml-[1.0625rem] checked:after:h-5 checked:after:w-5 checked:after:rounded-full checked:after:border-none checked:after:bg-blue-600 checked:after:shadow-[0_3px_1px_-2px_rgba(0,0,0,0.2),_0_2px_2px_0_rgba(0,0,0,0.14),_0_1px_5px_0_rgba(0,0,0,0.12)] checked:after:transition-[background-color_0.2s,transform_0.2s] checked:after:content-[''] hover:cursor-pointer focus:outline-none focus:ring-0 focus:before:scale-100 focus:before:opacity-[0.12] focus:before:shadow-[3px_-1px_0px_13px_rgba(0,0,0,0.6)] focus:before:transition-[box-shadow_0.2s,transform_0.2s] focus:after:absolute focus:after:z-[1] focus:after:block focus:after:h-5 focus:after:w-5 focus:after:rounded-full focus:after:content-[''] checked:focus:border-#0000ff checked:focus:bg-#0000ff checked:focus:before:ml-[1.0625rem] checked:focus:before:scale-100 checked:focus:before:shadow-[3px_-1px_0px_13px_#3b71ca] checked:focus:before:transition-[box-shadow_0.2s,transform_0.2s] dark:bg-neutral-600 dark:after:bg-neutral-500 dark:checked:bg-#0000ff dark:checked:after:bg-#0000ff] dark:focus:before:shadow-[3px_-1px_0px_13px_rgba(255,255,255,0.4)] dark:checked:focus:before:shadow-[3px_-1px_0px_13px_#3b71ca] ${
                      !rowStates[empresa.empresa_id] ? "active" : ""
                    }`}
                    type="checkbox"
                    role="switch"
                    id={`SwitchCheck_${empresa.empresa_id}`}
                    checked = {empresa.vigente}
                  />
                </td>

                <td className="py-2 px-4">
                  <input
                    className={`h-3.5 w-8  appearance-none rounded-[0.4375rem] bg-neutral-300 before:pointer-events-none before:absolute before:h-3.5 before:w-3.5 before:rounded-full before:bg-transparent before:content-[''] after:absolute after:z-[2] after:-mt-[0.1875rem] after:h-5 after:w-5  after:ml-[-0.2rem] after:rounded-full after:border-none after:bg-neutral-100 after:shadow-[0_0px_3px_0_rgb(0_0_0/_7%),_0_2px_2px_0_rgb(0_0_0/_4%)] after:transition-[background-color_0.2s,transform_0.2s] after:content-[''] checked:bg-blue-500 checked:after:absolute checked:after:z-[2] checked:after:-mt-[3px] checked:after:ml-[1.0625rem] checked:after:h-5 checked:after:w-5 checked:after:rounded-full checked:after:border-none checked:after:bg-blue-600 checked:after:shadow-[0_3px_1px_-2px_rgba(0,0,0,0.2),_0_2px_2px_0_rgba(0,0,0,0.14),_0_1px_5px_0_rgba(0,0,0,0.12)] checked:after:transition-[background-color_0.2s,transform_0.2s] checked:after:content-[''] hover:cursor-pointer focus:outline-none focus:ring-0 focus:before:scale-100 focus:before:opacity-[0.12] focus:before:shadow-[3px_-1px_0px_13px_rgba(0,0,0,0.6)] focus:before:transition-[box-shadow_0.2s,transform_0.2s] focus:after:absolute focus:after:z-[1] focus:after:block focus:after:h-5 focus:after:w-5 focus:after:rounded-full focus:after:content-[''] checked:focus:border-#0000ff checked:focus:bg-#0000ff checked:focus:before:ml-[1.0625rem] checked:focus:before:scale-100 checked:focus:before:shadow-[3px_-1px_0px_13px_#3b71ca] checked:focus:before:transition-[box-shadow_0.2s,transform_0.2s] dark:bg-neutral-600 dark:after:bg-neutral-500 dark:checked:bg-#0000ff dark:checked:after:bg-#0000ff] dark:focus:before:shadow-[3px_-1px_0px_13px_rgba(255,255,255,0.4)] dark:checked:focus:before:shadow-[3px_-1px_0px_13px_#3b71ca] ${
                      !rowStates[empresa.empresa_id] ? "active" : ""
                    }`}
                    type="checkbox"
                    role="switch"
                    id={`SwitchCheck2_${empresa.empresa_id}`}
                    checked = {empresa.postulaciones}
                  />
                </td>

                <td className="py-2 px-4">
                  <input
                    className={`h-3.5 w-8  appearance-none rounded-[0.4375rem] bg-neutral-300 before:pointer-events-none before:absolute before:h-3.5 before:w-3.5 before:rounded-full before:bg-transparent before:content-[''] after:absolute after:z-[2] after:-mt-[0.1875rem] after:h-5 after:w-5  after:ml-[-0.2rem] after:rounded-full after:border-none after:bg-neutral-100 after:shadow-[0_0px_3px_0_rgb(0_0_0/_7%),_0_2px_2px_0_rgb(0_0_0/_4%)] after:transition-[background-color_0.2s,transform_0.2s] after:content-[''] checked:bg-blue-500 checked:after:absolute checked:after:z-[2] checked:after:-mt-[3px] checked:after:ml-[1.0625rem] checked:after:h-5 checked:after:w-5 checked:after:rounded-full checked:after:border-none checked:after:bg-blue-600 checked:after:shadow-[0_3px_1px_-2px_rgba(0,0,0,0.2),_0_2px_2px_0_rgba(0,0,0,0.14),_0_1px_5px_0_rgba(0,0,0,0.12)] checked:after:transition-[background-color_0.2s,transform_0.2s] checked:after:content-[''] hover:cursor-pointer focus:outline-none focus:ring-0 focus:before:scale-100 focus:before:opacity-[0.12] focus:before:shadow-[3px_-1px_0px_13px_rgba(0,0,0,0.6)] focus:before:transition-[box-shadow_0.2s,transform_0.2s] focus:after:absolute focus:after:z-[1] focus:after:block focus:after:h-5 focus:after:w-5 focus:after:rounded-full focus:after:content-[''] checked:focus:border-#0000ff checked:focus:bg-#0000ff checked:focus:before:ml-[1.0625rem] checked:focus:before:scale-100 checked:focus:before:shadow-[3px_-1px_0px_13px_#3b71ca] checked:focus:before:transition-[box-shadow_0.2s,transform_0.2s] dark:bg-neutral-600 dark:after:bg-neutral-500 dark:checked:bg-#0000ff dark:checked:after:bg-#0000ff] dark:focus:before:shadow-[3px_-1px_0px_13px_rgba(255,255,255,0.4)] dark:checked:focus:before:shadow-[3px_-1px_0px_13px_#3b71ca] ${
                      !rowStates[empresa.empresa_id] ? "active" : ""
                    }`}
                    type="checkbox"
                    role="switch"
                    id={`SwitchCheck3_${empresa.empresa_id}`}
                    checked = {empresa.rubro}
                  />
                </td>

                <td className="py-2 px-4 text-center items-center">
              
                  <Link href={"/admin/dashboard/empresas/" + empresa.empresa_id} className="inline-block bg-blue-500 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded">
                    <DocumentIcon className="h-6 w-6 mx-auto text-white" />
                  </Link>
      
                  
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
      <div>
        {/* Mensaje de éxito */}
        {successMessage.visible && (
          <div className="bg-green-200 text-green-700 p-3 mt-2 rounded-md absolute bottom-4 right-3 w-[20%] h-20 justify-center text-center font-bold m-2 flex items-center">
            {successMessage.message}
          </div>
        )}
        {/* Modal */}
        <Modal isOpen={isModalOpen} onClose={closeModal}>
          {modalContent}
        </Modal>
      </div>
    </div>
  );
}
